/*
 * pwm_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
 
/******************************************************************************
 *
 * The vPWMTask task configures the PWM0 peripheral of the TM4C123GH6PM MCU to
 * output a PWM on PB6 that runs at 250 Hz with an initial duty cycle of 25%.
 * In order to get such a low speed PWM, the PWM clock is set to be the system
 * clock divided by 8.
 *
 * prvPWMInversionTask then delays for two second intervals before executing
 * the PWM inversion.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/pwm.h"
#include "driverlib/sysctl.h"
#include "drivers/rtos_hw_drivers.h"
/*-----------------------------------------------------------*/

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvPWMInversionTask( void *pvParameters );

/*
 * Called by main() to create the PWM task.
 */
void vPWMTask( void );

/* 
 * Configure the PWM 0 peripheral on the TM4C123GH6PM to output a 250 Hz PWM
 * at 25% duty cycle.
 */
static void prvConfigurePWM( void );
/*-----------------------------------------------------------*/

void vPWMTask( void )
{
    /* Configure the PWM module. */
    prvConfigurePWM();

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name PWM0 task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - No parameter passed to the task
     *  - The priority assigned to the task.
     *  - The task handle is NULL */
    xTaskCreate( prvPWMInversionTask,
                 "PWM0",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 NULL );
}

/*-----------------------------------------------------------*/

static void prvPWMInversionTask( void *pvParameters )
{
    for( ;; )
    {
        /* Create a two second delay. */
        vTaskDelay( pdMS_TO_TICKS( 2000 ) );

        /* Invert PWM0 signal. */
        PWMOutputInvert(PWM0_BASE, PWM_OUT_0_BIT, true);

        /* Create a two second delay. */
        vTaskDelay( pdMS_TO_TICKS( 2000 ) );

        /* Switch PWM0 signal back to regular operation. */
        PWMOutputInvert(PWM0_BASE, PWM_OUT_0_BIT, false);
    }
}
/*-----------------------------------------------------------*/

static void prvConfigurePWM( void )
{
    /* The PWM peripheral must be enabled for use. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_PWM0);

    /* Configure the PWM function for this pin. */
    GPIOPinConfigure(GPIO_PB6_M0PWM0);
    GPIOPinTypePWM(GPIO_PORTB_BASE, GPIO_PIN_6);

    /* Set the PWM clock to be SysClk / 8. */
    SysCtlPWMClockSet(SYSCTL_PWMDIV_8);

    /* Configure PWM0 to count up/down without synchronization. */
    PWMGenConfigure(PWM0_BASE, PWM_GEN_0, PWM_GEN_MODE_UP_DOWN |
                    PWM_GEN_MODE_NO_SYNC);

    /* Set the PWM period to 250Hz.  To calculate the appropriate parameter use
     * the following equation: N = (1 / f) * PWMClk.  Where N is the function
     * parameter, f is the desired frequency, and PWMClk is the PWM clock
     * frequency that is derived from the system clock frequency divided by the
     * PWM clock divider used (if any). */
    PWMGenPeriodSet(PWM0_BASE, PWM_GEN_0, ( (configCPU_CLOCK_HZ / 8 ) / 250) );

    /* Set PWM0 to a duty cycle of 25%.  You set the duty cycle as a function
     * of the period.  Since the period was set above, you can use the
     * PWMGenPeriodGet() function.  For this example the PWM will be high for
     * 25% of the time or (PWM Period / 4). */
    PWMPulseWidthSet(PWM0_BASE, PWM_OUT_0,
                     PWMGenPeriodGet(PWM0_BASE, PWM_GEN_0) / 4);

    /* Enable PWM Out Bit 0 (PB6) output signal. */
    PWMOutputState(PWM0_BASE, PWM_OUT_0_BIT, true);

    /* Enable the PWM generator block. */
    PWMGenEnable(PWM0_BASE, PWM_GEN_0);
}
/*-----------------------------------------------------------*/

void vApplicationTickHook( void )
{
    /* This function will be called by each tick interrupt if
        configUSE_TICK_HOOK is set to 1 in FreeRTOSConfig.h.  User code can be
        added here, but the tick hook is called from an interrupt context, so
        code must not attempt to block, and only the interrupt safe FreeRTOS API
        functions can be used (those that end in FromISR()). */

    /* Only the full demo uses the tick hook so there is no code is
        executed here. */
}


